<?php
/**
 * Agency Taxonomy Filters for Admin List
 *
 * This file implements admin-side filtering capabilities for the 'estate_agency' custom post type.
 * It adds dropdown filter controls in the WordPress admin list view that allow administrators
 * to filter agency listings by various taxonomies (categories, action categories, and cities).
 * The implementation uses closures to create reusable filter functions that can be applied
 * to different taxonomy and post type combinations.
 *
 * @package WPResidence
 * @subpackage AdminFilters
 * @version 1.0
 * @author WPResidence Team
 * @copyright Copyright (c) WPResidence
 * @license GPL2+
 */

/**
 * Creates a closure that generates a dropdown filter for a specific post type and taxonomy
 *
 * This function returns another function (closure) configured for a specific post type
 * and taxonomy combination. When the returned function is called by WordPress hooks,
 * it displays a dropdown filter in the admin posts list if the current post type matches.
 * 
 * The dropdown shows all terms from the specified taxonomy with post counts, allowing
 * administrators to filter the list by selecting a term.
 *
 * @since 1.0
 * @param string $post_type The post type to check against current screen
 * @param string $taxonomy The taxonomy whose terms will populate the dropdown
 * @return callable A function to be used as a callback for restrict_manage_posts hook
 */
$restrict_manage_posts = function($post_type, $taxonomy) {
    return function() use($post_type, $taxonomy) {
        global $typenow;
        // Only show the filter on the specified post type screen
        if($typenow == $post_type) {
            // Get the currently selected term, if any
            $selected = isset($_GET[$taxonomy]) ? $_GET[$taxonomy] : '';

            if (taxonomy_exists($taxonomy)) {
                // Get taxonomy object to access its labels
                $info_taxonomy = get_taxonomy($taxonomy);

                // Generate the dropdown with all taxonomy terms
                wp_dropdown_categories(array(
                    'show_option_all'   => esc_html__("Show All {$info_taxonomy->label}", 'wpresidence-core'),
                    'taxonomy'          => $taxonomy,
                    'name'              => $taxonomy,
                    'orderby'           => 'name',
                    'selected'          => $selected,
                    'show_count'        => TRUE,
                    'hide_empty'        => TRUE,
                    'hierarchical'      => true
                ));
            }
        }
    };
};

/**
 * Creates a closure that converts taxonomy term IDs to slugs in admin queries
 *
 * This function returns a closure configured for a specific post type and taxonomy.
 * The closure modifies WordPress queries on the edit.php admin page to convert
 * numeric taxonomy term IDs into term slugs, which is necessary because WordPress
 * uses term slugs (not IDs) for taxonomy queries.
 *
 * When a user selects a term from the dropdown generated by $restrict_manage_posts,
 * this function ensures the query properly filters posts by the selected term.
 *
 * @since 1.0
 * @param string $post_type The post type to check against in the query
 * @param string $taxonomy The taxonomy to process term conversion for
 * @return callable A function to be used as a callback for parse_query filter
 */
$parse_query = function($post_type, $taxonomy) {
    return function($query) use($post_type, $taxonomy) {
        global $pagenow;
        $q_vars = &$query->query_vars;
        // Only modify queries on the edit.php page for our specific post type
        // and only when a numeric taxonomy filter is applied
        if( $pagenow == 'edit.php'
            && isset($q_vars['post_type']) && $q_vars['post_type'] == $post_type
            && isset($q_vars[$taxonomy])
            && is_numeric($q_vars[$taxonomy]) && $q_vars[$taxonomy] != 0
        ) {
            // Convert the term ID to a term slug for proper querying
            $term = get_term_by('id', $q_vars[$taxonomy], $taxonomy);
            $q_vars[$taxonomy] = $term->slug;
        }
    };
};

/**
 * Hook registrations for Agency Category taxonomy filters
 * 
 * Adds dropdown filter and query modification for the 'category_agency' taxonomy
 * on the 'estate_agency' post type admin list.
 */
add_action('restrict_manage_posts', $restrict_manage_posts('estate_agency', 'category_agency') );
add_filter('parse_query', $parse_query('estate_agency', 'category_agency') );

/**
 * Hook registrations for Agency Action Category taxonomy filters
 * 
 * Adds dropdown filter and query modification for the 'action_category_agency' taxonomy
 * on the 'estate_agency' post type admin list.
 */
add_action('restrict_manage_posts', $restrict_manage_posts('estate_agency', 'action_category_agency') );
add_filter('parse_query', $parse_query('estate_agency', 'action_category_agency') );

/**
 * Hook registrations for Agency City taxonomy filters
 * 
 * Adds dropdown filter and query modification for the 'city_agency' taxonomy
 * on the 'estate_agency' post type admin list.
 */
add_action('restrict_manage_posts', $restrict_manage_posts('estate_agency', 'city_agency') );
add_filter('parse_query', $parse_query('estate_agency', 'city_agency') );